﻿using System;
using System.Data;
using System.Globalization;
using System.Text;
using NHibernate.Dialect;
using NHibernate.SqlTypes;
using NHibernate.Type;

namespace HelloEncryptedFluent.Encrypted
{
    public class EncryptedType<T> : PrimitiveType
    {
        private readonly IEncryptionService encrypter;
        private readonly IBinaryConverter converter;

        public EncryptedType() : base(new BinarySqlType(Int32.MaxValue))
         {
             this.encrypter = new EncryptionService();
             this.converter = new BinaryConverter();
         }

        public override string Name
        {
            get { return typeof(T).Name; }
        }

        public override Type ReturnedClass
        {
            get { return typeof(T); }
        }

        public override Type PrimitiveClass
        {
            get { return typeof(T); }
        }

        public override object DefaultValue
        {
            get { return default(T); }
        }

        public override void Set(IDbCommand cmd, object value, int index)
        {
            var serialized = this.converter.Serialize(value);
            var encrypted = this.encrypter.Encrypt(serialized);
            ((IDataParameter)cmd.Parameters[index]).Value = encrypted;
        }

        public override object Get(IDataReader rs, int index)
        {
            if (rs.IsDBNull(index))
                return null;

            var encrypted = rs[index] as byte[];

            var decrypted = this.encrypter.Decrypt(encrypted);
            var deserialized = this.converter.Deserialize(decrypted);

            return deserialized;
        }

        public override object Get(IDataReader rs, string name)
        {
            return this.Get(rs, rs.GetOrdinal(name));
        }

        public override object FromStringValue(string xml)
        {
            if (xml == null)
                return null;

            if (xml.Length % 2 != 0)
                throw new ArgumentException("The string is not a valid xml representation of a binary content.", "xml");

            var bytes = new byte[xml.Length / 2];
            for (var i = 0; i < bytes.Length; i++)
            {
                var hexStr = xml.Substring(i * 2, (i + 1) * 2);
                bytes[i] = (byte)(byte.MinValue + byte.Parse(hexStr, NumberStyles.HexNumber, CultureInfo.InvariantCulture));
            }

            var decrypted = this.encrypter.Decrypt(bytes);
            var deserialized = this.converter.Deserialize(decrypted);
            return deserialized;
        }

        public override string ObjectToSQLString(object value, Dialect dialect)
        {
            var bytes = value as byte[];
            if (bytes == null)
                return "NULL";
            var builder = new StringBuilder();
            for (int i = 0; i < bytes.Length; i++)
            {
                string hexStr = (bytes[i] - byte.MinValue).ToString("x", CultureInfo.InvariantCulture);
                if (hexStr.Length == 1)
                    builder.Append('0');
                builder.Append(hexStr);
            }
            return builder.ToString();
        }

    }       

}
